//
//  YHButlerServiceManager.swift
//  galaxy
//
//  Created by alexzzw on 2024/9/4.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import IQKeyboardManagerSwift

class YHButlerServiceManager: NSObject {
    
    static let shared = YHButlerServiceManager()
    
    private weak var conversationManager = QYSDK.shared().conversationManager()
    
    // 调整UI
    private weak var customUIConfig = QYSDK.shared().customUIConfig()
    
    private lazy var getUserInfoViewModel: YHButlerServiceViewModel = YHButlerServiceViewModel()
    
    var lastMessage: YHButlerServiceMessage?
    var unreadCount: Int {
        allUnreadCount()
    }
        
    private override init() {
        super.init()
        conversationManager?.setDelegate(self)
        
        NotificationCenter.default.addObserver(self, selector: #selector(loginSuccess), name: YhConstant.YhNotification.didLoginSuccessNotifiction, object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(logoutSuccess), name: YhConstant.YhNotification.didLogoutSuccessNotifiction, object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(didRevUatParam), name: YhConstant.YhNotification.didRevUatParam, object: nil)
    }
    
    deinit {
        NotificationCenter.default.removeObserver(self)
    }
        
    func setupSDK() {
        let qyOption = QYSDKOption(appKey: YhConstant.QiYuSDK.appKey)
        qyOption.appName = YhConstant.QiYuSDK.appName
        QYSDK.shared().register(with: qyOption)
        /*
        QYSDK.shared().registerPushMessageNotification { pushMessage in
            os_log("#####registerPushMessageNotification %@ %ld %f", (pushMessage?.text ?? ""), (pushMessage?.type.rawValue ?? 0), (pushMessage?.time ?? 0))
            printLog("$$$$registerPushMessageNotification \(pushMessage?.text ?? "") \(pushMessage?.type.rawValue ?? 0) \(pushMessage?.time ?? 0)")
        }
        */
        guard isUatAllowed() else {
            return
        }
        initSDKAccountAndLastMessage()
    }
            
    func updateApnsToken(token: Data) {
        QYSDK.shared().updateApnsToken(token)
    }
    
    @MainActor func gotoButlerServiceSessionVC(sourceTitle: String? = nil, sourceUrl: String? = nil, sessionTitle: String = "银河管家", navigationController: UINavigationController?, completion: @escaping (Bool) -> Void) {
        
        func gotoSessionVC() {
            let source = QYSource()
            source.title = sourceTitle
            source.urlString = sourceUrl
            IQKeyboardManager.shared.disabledDistanceHandlingClasses.append(QYSessionViewController.self)
            guard let sessionViewController = QYSDK.shared().sessionViewController() else {
                completion(false)
                return
            }
            completion(true)
            sessionViewController.sessionTitle = sessionTitle
            sessionViewController.source = source
            sessionViewController.hidesBottomBarWhenPushed = true
            navigationController?.pushViewController(sessionViewController)
        }
        
        guard let userId = YHLoginManager.shared.userModel?.id, userId.count > 0 else {
            completion(false)
            return
        }
        getUserInfoViewModel.getUserInfomation(userId) { [weak self] dataString, avatar, error in
            // 改变头像
            self?.customUIConfig?.customerHeadImageUrl = avatar
            self?.updateUserInfo(userId: userId, userData: dataString ?? "", completion: { success in
                gotoSessionVC()
            })
        }
    }
}


extension YHButlerServiceManager {
    // MARK: - 常用公开方法
    
    /// 灰度打开
    func isUatAllowed() -> Bool {
        let allowed = YHUatHelper.shared.getUatModuleStatusBy(module: .QiYu_Service) == 1
        //TODO: for test Alex  
        return allowed || true
    }
    
    /// 初始化SDK用户信息
    func initSDKAccountAndLastMessage() {
        updateSDKAccountInfo(completion: { [weak self] _ in
            self?.lastMessage = self?.getLastMessage()
        })
    }
        
    /// 必须在初始化后调用
    func getLastMessage() -> YHButlerServiceMessage? {
        guard let message = getSessionList().last else {
            return nil
        }
        return YHButlerServiceMessageHandler.handleSessionMessage(sessionMessage: message)
    }
}

extension YHButlerServiceManager {
    // MARK: - 私有方法以及登录登出的处理逻辑
    
    @objc private func didRevUatParam() {
        guard isUatAllowed() else {
            return
        }
        initSDKAccountAndLastMessage()
    }
    
    @objc private func loginSuccess() {
        guard isUatAllowed() else {
            return
        }
        initSDKAccountAndLastMessage()
    }
    
    @objc private func logoutSuccess() {
        // 如果登出时需要关闭聊天页
        /*
        if let controller = QYSDK.shared().sessionViewController(), controller.isViewLoaded && controller.view.window != nil  {
            controller.closeSession(true) { _, _ in
                //
            }
        }
        */
        logoutCurrentUser { _ in
        }
    }
    
    private func updateUserInfo(userId: String, userData: String, completion: @escaping (Bool) -> Void) {
        let userInfo = QYUserInfo()
        userInfo.userId = userId
        userInfo.data = userData
        QYSDK.shared().setUserInfo(userInfo) { success, error in
            completion(success)
        }
    }
    
    private func setupInfomationForQiYu(completion: @escaping (Bool) -> Void) {
        guard let userId = YHLoginManager.shared.userModel?.id, userId.count > 0 else {
            completion(false)
            return
        }
        getUserInfoViewModel.getUserInfomation(userId) { [weak self] dataString, avatar, error in
            // 改变头像
            self?.customUIConfig?.customerHeadImageUrl = avatar
            self?.updateUserInfo(userId: userId, userData: dataString ?? "", completion: { success in
                completion(success)
            })
        }
    }
    
    /// 启动先判断是否当前登录账号是SDK账号
    private func updateSDKAccountInfo(completion: @escaping (Bool) -> Void) {
        // 如果未登录直接退出设置
        guard let userId = YHLoginManager.shared.userModel?.id, userId.count > 0 else {
            completion(false)
            return
        }
        // 如果新旧账号相同，直接设置用户账号信息
        guard QYSDK.shared().currentUserID() != userId else {
            setupInfomationForQiYu(completion: { done in
                completion(done)
            })
            return
        }
        // 如果新旧账号不相同，先登出SDK，再用新账号信息写入，
        logoutCurrentUser { [weak self] success in
            if success {
                self?.setupInfomationForQiYu(completion: { done in
                    completion(done)
                })
            } else {
                completion(success)
            }
        }
    }
    
    private func logoutCurrentUser(completion: @escaping (Bool) -> Void) {
        QYSDK.shared().logout { flag in
            completion(flag)
        }
    }
}

extension YHButlerServiceManager {
    // MARK: - conversationManager方法
    
    /// 获取所有会话的列表；非平台电商用户，只有一个会话项，平台电商用户，有多个会话项
    func getSessionList() -> [QYSessionInfo] {
        let list = conversationManager?.getSessionList() ?? []
        printLog("$$$$getSessionList \(list.last?.lastMessageText ?? "")  \(list.last?.lastMessageType.rawValue ?? 0)  \(list.last?.lastMessageTimeStamp ?? 0)")
        return list
    }
    
    /// 所有的未读数
    func allUnreadCount() -> Int {
        let count = conversationManager?.allUnreadCount() ?? 0
        printLog("$$$$allUnreadCount \(count)")
        return count
    }
    
    /// 清空未读数
    func clearUnreadCount() {
        conversationManager?.clearUnreadCount()
        // 清空已读消息后刷新最新消息状态
        lastMessage = getLastMessage()
        printLog("$$$$clearUnreadCount")
        //unreadCount = 0
        NotificationCenter.default.post(name: YhConstant.YhNotification.didQiYuUnReadMsgCountChangeNotification, object: nil)
    }
    
}

extension YHButlerServiceManager: QYConversationManagerDelegate {
    // MARK: - QYConversationManagerDelegate
    
    /// 会话列表变化；非平台电商用户，只有一个会话项，平台电商用户，有多个会话项  用这个方法可以拿到最新的完整的消息内容
    func onSessionListChanged(_ sessionList: [QYSessionInfo]!) {
        printLog("$$$$onSessionListChanged \(sessionList.first?.lastMessageText ?? "") \(sessionList.first?.lastMessageType.rawValue ?? 0) \(sessionList.first?.lastMessageTimeStamp ?? 0) \(sessionList.first?.lastMessage?.messageId ?? "")")
        guard let message = sessionList.first else {
            return
        }
        lastMessage = YHButlerServiceMessageHandler.handleSessionMessage(sessionMessage: message)
        NotificationCenter.default.post(name: YhConstant.YhNotification.didQiYuReceiveNewMsgNotification, object: nil)
    }
    
    /// 接收消息（进聊天页面会刷新， 目前不用这个方法打点记录）
    func onReceiveMessage(_ message: QYMessageInfo!) {
        printLog("$$$$onReceiveMessage \(message?.text ?? "") \(message?.type.rawValue ?? 0) \(message?.timeStamp ?? 0)")
    }
    
    /// 会话未读数变化（进聊天页面会刷新）
    func onUnreadCountChanged(_ count: Int) {
        printLog("$$$$onUnreadCountChanged \(count)")
        //unreadCount = count
        NotificationCenter.default.post(name: YhConstant.YhNotification.didQiYuUnReadMsgCountChangeNotification, object: nil)
    }
}
