//
//  YHMajorSearchViewController.swift
//  galaxy
//
//  Created by Dufet on 2024/11/19.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHMajorSearchViewController: YHBaseViewController {
    
    var orderId: Int = 0
    var searchMajorName: String?
    var selectBlock: ((String?, String?) -> Void)?
    var majors: [YHMajorInfo] = []
    let educationRequest: YHEducationRequestViewModel = YHEducationRequestViewModel()

    lazy var searchBar: YHCollegeSearchBar = {
        let bar = YHCollegeSearchBar.createBar()
        bar.placeHolder = "请输入专业"
        bar.textField.clearButtonMode = .whileEditing
        return bar
    }()
    
    lazy var tableView: UITableView = {
        
        let tableView = UITableView(frame: .zero, style: .plain)
        
        if #available(iOS 11.0, *) {
            tableView.contentInsetAdjustmentBehavior = .never
        }
        tableView.showsVerticalScrollIndicator = false
        tableView.backgroundColor = .white
        tableView.separatorStyle = .none
        tableView.delegate = self
        tableView.dataSource = self
        tableView.register(UITableViewCell.self, forCellReuseIdentifier: "UITableViewCell")
        tableView.register(YHMajorNameCell.self, forCellReuseIdentifier: YHMajorNameCell.cellReuseIdentifier)
        tableView.backgroundView = emptyDataTipsView
        return tableView
    }()
    
    lazy var emptyDataTipsView: YHEmptyDataView = {
        let view = YHEmptyDataView.createView("无搜索结果", kEmptySearchBgName)
        view.topMargin = 130
        view.isHidden = true
        return view
    }()

    override func viewDidLoad() {
        super.viewDidLoad()
        createUI()
        if searchMajorName != nil {
            searchMajor()
        }
    }
    
    func createUI() {
        
        self.view.backgroundColor = .white
        self.gk_navTitle = "专业名称".local

        view.addSubview(tableView)
        view.addSubview(searchBar)
        
        if let searchMajorName = searchMajorName {
            searchBar.textField.text = searchMajorName
        }
        searchBar.textChange = {
           [weak self]  text in
            guard let self = self else { return }
            searchMajorName = text
            self.searchMajor()
        }
        searchBar.confirmBlock = {
            [weak self] in
            guard let self = self else { return }
            if let selectBlock = selectBlock {
                selectBlock(searchBar.textField.text, searchBar.textField.text)
                self.navigationController?.popViewController(animated: true)
            }
        }
        
        searchBar.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.right.equalToSuperview().offset(-16)
            make.height.equalTo(YHCollegeSearchBar.height)
            make.top.equalToSuperview().offset(k_Height_NavigationtBarAndStatuBar+16)
        }
        
        tableView.snp.makeConstraints { make in
            make.top.equalTo(searchBar.snp.bottom).offset(8)
            make.left.right.bottom.equalToSuperview()
        }
    }
    
    func searchMajor() {
        
        let param: [String: Any] = ["order_id": orderId,
                                  "name": searchMajorName ?? "",
                                  "page": 1,
                                  "page_size": 15
                                 ]
        self.educationRequest.searchMajorInfoList(params: param) {
            [weak self] success, _ in
            guard let self = self else { return }
            if success {
                if let list = self.educationRequest.majorList {
                    majors = list
                } else {
                    majors = []
                }
                emptyDataTipsView.isHidden = majors.count > 0
            }
            self.tableView.reloadData()
        }
    }
}

extension YHMajorSearchViewController: UITableViewDelegate, UITableViewDataSource {
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return majors.count
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return UITableView.automaticDimension
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        guard let cell = tableView.dequeueReusableCell(withIdentifier: YHMajorNameCell.cellReuseIdentifier, for: indexPath) as? YHMajorNameCell else {
            return UITableViewCell()
        }
        if indexPath.row < majors.count {
            let college: YHMajorInfo = majors[indexPath.row]
            
            var name = ""
            if !isEmptyString(college.ch_name) {
                name += college.ch_name
            }
            if !isEmptyString(college.en_name) && !isEmptyString(college.en_name) {
                name += "/"
            }
            if !isEmptyString(college.en_name) {
                name += college.en_name
            }
            cell.titleLabel.text = name
        }
        return cell
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        if let selectBlock = selectBlock {
            if indexPath.row < majors.count {
                let major: YHMajorInfo = majors[indexPath.row]
                selectBlock(major.getMajorName(), major.ch_name)
                self.navigationController?.popViewController(animated: true)
            }
        }
    }
}
