//
//  LBXScanWrapper.swift
//  swiftScan
//
//  Created by lbxia on 15/12/10.
//  Copyright © 2015年 xialibing. All rights reserved.
//

import AVFoundation
import UIKit

public struct LBXScanResult {
    // 码内容
    public var strScanned: String? = ""
    // 扫码图像
    public var imgScanned: UIImage?
    // 码的类型
    public var strBarCodeType: String? = ""

    // 码在图像中的位置
    public var arrayCorner: [AnyObject]?

    public init(str: String?, img: UIImage?, barCodeType: String?, corner: [AnyObject]?) {
        strScanned = str
        imgScanned = img
        strBarCodeType = barCodeType
        arrayCorner = corner
    }
}

open class LBXScanWrapper: NSObject, AVCaptureMetadataOutputObjectsDelegate {
    let device = AVCaptureDevice.default(for: AVMediaType.video)
    var input: AVCaptureDeviceInput?
    var output: AVCaptureMetadataOutput

    let session = AVCaptureSession()
    var previewLayer: AVCaptureVideoPreviewLayer?
    var stillImageOutput: AVCaptureStillImageOutput?

    // 存储返回结果
    var arrayResult: [LBXScanResult] = []

    // 扫码结果返回block
    var successBlock: ([LBXScanResult]) -> Void

    // 是否需要拍照
    var isNeedCaptureImage: Bool

    // 当前扫码结果是否处理
    var isNeedScanResult: Bool = true

    private let sessionQueue = DispatchQueue(label: "com.lbxscan.session", qos: .userInitiated)

    init(videoPreView: UIView, objType: [AVMetadataObject.ObjectType] = [.qr], isCaptureImg: Bool, cropRect: CGRect = CGRect.zero, success: @escaping (([LBXScanResult]) -> Void)) {
        successBlock = success
        output = AVCaptureMetadataOutput()
        isNeedCaptureImage = isCaptureImg
        stillImageOutput = AVCaptureStillImageOutput()

        super.init()

        guard let device = device else { return }

        do {
            input = try AVCaptureDeviceInput(device: device)
        } catch let error as NSError {
            print("AVCaptureDeviceInput(): \(error)")
            return
        }

        guard let validInput = input else { return }

        if session.canAddInput(validInput) {
            session.addInput(validInput)
        }

        if session.canAddOutput(output) {
            session.addOutput(output)
        }

        if let stillOutput = stillImageOutput, session.canAddOutput(stillOutput) {
            session.addOutput(stillOutput)
        }

        let outputSettings: [String: Any] = [AVVideoCodecKey: AVVideoCodecType.jpeg]
        stillImageOutput?.outputSettings = outputSettings

        session.sessionPreset = AVCaptureSession.Preset.high

        // 参数设置
        output.setMetadataObjectsDelegate(self, queue: DispatchQueue.main)

        output.metadataObjectTypes = objType

        if !cropRect.equalTo(CGRect.zero) {
            // 启动相机后，直接修改该参数无效
            output.rectOfInterest = cropRect
        }

        previewLayer = AVCaptureVideoPreviewLayer(session: session)
        previewLayer?.videoGravity = AVLayerVideoGravity.resizeAspectFill

        var frame: CGRect = videoPreView.frame
        frame.origin = CGPoint.zero
        previewLayer?.frame = frame

        if let layer = previewLayer {
            videoPreView.layer.insertSublayer(layer, at: 0)
        }

        if device.isFocusPointOfInterestSupported && device.isFocusModeSupported(AVCaptureDevice.FocusMode.continuousAutoFocus) {
            do {
                try device.lockForConfiguration()

                device.focusMode = AVCaptureDevice.FocusMode.continuousAutoFocus

                device.unlockForConfiguration()
            } catch let error as NSError {
                print("device.lockForConfiguration(): \(error)")
            }
        }
    }

    public func metadataOutput(_ output: AVCaptureMetadataOutput, didOutput metadataObjects: [AVMetadataObject], from connection: AVCaptureConnection) {
        if !isNeedScanResult {
            // 上一帧处理中
            return
        }

        isNeedScanResult = false

        arrayResult.removeAll()

        // 识别扫码类型
        for current in metadataObjects {
            if let code = current as? AVMetadataMachineReadableCodeObject {
                // 码类型
                let codeType = code.type
                // 码内容
                let codeContent = code.stringValue

                arrayResult.append(LBXScanResult(str: codeContent, img: UIImage(), barCodeType: codeType.rawValue, corner: code.corners as [AnyObject]?))
            }
        }

        if arrayResult.count > 0 {
            if isNeedCaptureImage {
                captureImage()
            } else {
                stop()
                successBlock(arrayResult)
            }

        } else {
            isNeedScanResult = true
        }
    }

    func start() {
        sessionQueue.async {
            if !self.session.isRunning {
                self.isNeedScanResult = true
                self.session.startRunning()
            }
        }
    }

    func stop() {
        sessionQueue.async {
            if self.session.isRunning {
                self.isNeedScanResult = false
                self.session.stopRunning()
            }
        }
    }

    // MARK: - ---拍照

    open func captureImage() {
        guard let stillOutput = stillImageOutput else {
            stop()
            successBlock(arrayResult)
            return
        }

        let stillImageConnection: AVCaptureConnection? = connectionWithMediaType(mediaType: AVMediaType.video as AVMediaType, connections: stillOutput.connections as [AnyObject])

        guard let connection = stillImageConnection else {
            stop()
            successBlock(arrayResult)
            return
        }

        stillOutput.captureStillImageAsynchronously(from: connection, completionHandler: { imageDataSampleBuffer, _ in

            self.stop()
            if let sampleBuffer = imageDataSampleBuffer, let imageData: Data = AVCaptureStillImageOutput.jpegStillImageNSDataRepresentation(sampleBuffer) {
                let scanImg: UIImage? = UIImage(data: imageData)

                for idx in 0 ... self.arrayResult.count - 1 {
                    self.arrayResult[idx].imgScanned = scanImg
                }
            }

            self.successBlock(self.arrayResult)

        })
    }

    open func connectionWithMediaType(mediaType: AVMediaType, connections: [AnyObject]) -> AVCaptureConnection? {
        for connection in connections {
            guard let connectionTmp = connection as? AVCaptureConnection else { continue }

            for port in connectionTmp.inputPorts {
                if port.mediaType == mediaType {
                    return connectionTmp
                }
            }
        }
        return nil
    }

    // MARK: 切换识别区域

    open func changeScanRect(cropRect: CGRect) {
        sessionQueue.async {
            if self.session.isRunning {
                self.session.stopRunning()
            }

            DispatchQueue.main.async {
                self.output.rectOfInterest = cropRect

                self.sessionQueue.async {
                    self.session.startRunning()
                }
            }
        }
    }

    // MARK: 切换识别码的类型

    open func changeScanType(objType: [AVMetadataObject.ObjectType]) {
        // 待测试中途修改是否有效
        output.metadataObjectTypes = objType
    }

    open func isGetFlash() -> Bool {
        guard let device = device else { return false }
        return device.hasFlash && device.hasTorch
    }

    /**
     打开或关闭闪关灯
     - parameter torch: true：打开闪关灯 false:关闭闪光灯
     */
    open func setTorch(torch: Bool) {
        if isGetFlash() {
            guard let device = input?.device else { return }
            do {
                try device.lockForConfiguration()

                device.torchMode = torch ? AVCaptureDevice.TorchMode.on : AVCaptureDevice.TorchMode.off

                device.unlockForConfiguration()
            } catch let error as NSError {
                print("device.lockForConfiguration(): \(error)")
            }
        }
    }

    /**
     ------闪光灯打开或关闭
     */
    open func changeTorch() {
        if isGetFlash() {
            guard let device = input?.device else { return }
            do {
                try device.lockForConfiguration()

                var torch = false

                if device.torchMode == AVCaptureDevice.TorchMode.on {
                    torch = false
                } else if device.torchMode == AVCaptureDevice.TorchMode.off {
                    torch = true
                }

                device.torchMode = torch ? AVCaptureDevice.TorchMode.on : AVCaptureDevice.TorchMode.off

                device.unlockForConfiguration()
            } catch let error as NSError {
                print("device.lockForConfiguration(): \(error)")
            }
        }
    }

    // MARK: - -----获取系统默认支持的码的类型

    static func defaultMetaDataObjectTypes() -> [AVMetadataObject.ObjectType] {
        var types =
            [AVMetadataObject.ObjectType.qr,
             AVMetadataObject.ObjectType.upce,
             AVMetadataObject.ObjectType.code39,
             AVMetadataObject.ObjectType.code39Mod43,
             AVMetadataObject.ObjectType.ean13,
             AVMetadataObject.ObjectType.ean8,
             AVMetadataObject.ObjectType.code93,
             AVMetadataObject.ObjectType.code128,
             AVMetadataObject.ObjectType.pdf417,
             AVMetadataObject.ObjectType.aztec
            ]

        types.append(AVMetadataObject.ObjectType.interleaved2of5)
        types.append(AVMetadataObject.ObjectType.itf14)
        types.append(AVMetadataObject.ObjectType.dataMatrix)
        return types as [AVMetadataObject.ObjectType]
    }

    static func isSysIos8Later() -> Bool {
        if #available(iOS 8, *) {
            return true
        }
        return false
    }

    /**
     识别二维码码图像

     - parameter image: 二维码图像

     - returns: 返回识别结果
     */
    public static func recognizeQRImage(image: UIImage) -> [LBXScanResult] {
        var returnResult: [LBXScanResult] = []

        if LBXScanWrapper.isSysIos8Later() {
            guard let detector = CIDetector(ofType: CIDetectorTypeQRCode, context: nil, options: [CIDetectorAccuracy: CIDetectorAccuracyHigh]),
                  let cgImage = image.cgImage else {
                return returnResult
            }

            let img = CIImage(cgImage: cgImage)

            let features: [CIFeature]? = detector.features(in: img, options: [CIDetectorAccuracy: CIDetectorAccuracyHigh])

            if let validFeatures = features, validFeatures.count > 0 {
                let feature = validFeatures[0]

                if let featureTmp = feature as? CIQRCodeFeature {
                    let scanResult = featureTmp.messageString

                    let result = LBXScanResult(str: scanResult, img: image, barCodeType: AVMetadataObject.ObjectType.qr.rawValue, corner: nil)

                    returnResult.append(result)
                }
            }
        }

        return returnResult
    }

    // MARK: - - - 生成二维码，背景色及二维码颜色设置

    public static func createCode(codeType: String, codeString: String, size: CGSize, qrColor: UIColor, bkColor: UIColor) -> UIImage? {
        guard let stringData = codeString.data(using: String.Encoding.utf8),
              let qrFilter = CIFilter(name: codeType) else {
            return nil
        }

        qrFilter.setValue(stringData, forKey: "inputMessage")
        qrFilter.setValue("M", forKey: "inputCorrectionLevel")

        guard let outputImage = qrFilter.outputImage,
              let colorFilter = CIFilter(name: "CIFalseColor", parameters: ["inputImage": outputImage, "inputColor0": CIColor(cgColor: qrColor.cgColor), "inputColor1": CIColor(cgColor: bkColor.cgColor)]),
              let qrImage = colorFilter.outputImage,
              let cgImage = CIContext().createCGImage(qrImage, from: qrImage.extent) else {
            return nil
        }

        UIGraphicsBeginImageContext(size)
        guard let context = UIGraphicsGetCurrentContext() else {
            UIGraphicsEndImageContext()
            return nil
        }
        context.interpolationQuality = CGInterpolationQuality.none
        context.scaleBy(x: 1.0, y: -1.0)
        context.draw(cgImage, in: context.boundingBoxOfClipPath)
        let codeImage = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()

        return codeImage
    }

    public static func createCode128(codeString: String, size: CGSize, qrColor: UIColor, bkColor: UIColor) -> UIImage? {
        guard let stringData = codeString.data(using: String.Encoding.utf8),
              let qrFilter = CIFilter(name: "CICode39BarcodeGenerator") else {
            return nil
        }

        qrFilter.setDefaults()
        qrFilter.setValue(stringData, forKey: "inputMessage")

        guard let outputImage = qrFilter.outputImage else { return nil }

        let context = CIContext()
        guard let cgImage = context.createCGImage(outputImage, from: outputImage.extent) else { return nil }

        let image = UIImage(cgImage: cgImage, scale: 1.0, orientation: .up)

        // Resize without interpolating
        let scaleRate: CGFloat = 20.0
        let resized = resizeImage(image: image, quality: CGInterpolationQuality.none, rate: scaleRate)

        return resized
    }

    // MARK: 根据扫码结果，获取图像中得二维码区域图像（如果相机拍摄角度故意很倾斜，获取的图像效果很差）

    static func getConcreteCodeImage(srcCodeImage: UIImage, codeResult: LBXScanResult) -> UIImage? {
        let rect: CGRect = getConcreteCodeRectFromImage(srcCodeImage: srcCodeImage, codeResult: codeResult)

        if rect.isEmpty {
            return nil
        }

        let img = imageByCroppingWithStyle(srcImg: srcCodeImage, rect: rect)

        if let validImg = img {
            let imgRotation = imageRotation(image: validImg, orientation: .right)
            return imgRotation
        }
        return nil
    }

    // 根据二维码的区域截取二维码区域图像
    public static func getConcreteCodeImage(srcCodeImage: UIImage, rect: CGRect) -> UIImage? {
        if rect.isEmpty {
            return nil
        }

        let img = imageByCroppingWithStyle(srcImg: srcCodeImage, rect: rect)

        if let validImg = img {
            let imgRotation = imageRotation(image: validImg, orientation: .right)
            return imgRotation
        }
        return nil
    }

    // 获取二维码的图像区域
    public static func getConcreteCodeRectFromImage(srcCodeImage: UIImage, codeResult: LBXScanResult) -> CGRect {
        guard let corners = codeResult.arrayCorner, corners.count >= 4 else {
            return CGRect.zero
        }

        guard let corner = corners as? [[String: Float]] else {
            return CGRect.zero
        }

        let dicTopLeft = corner[0]
        let dicTopRight = corner[1]
        let dicBottomRight = corner[2]
        let dicBottomLeft = corner[3]

        guard let xLeftTopRatio = dicTopLeft["X"],
              let yLeftTopRatio = dicTopLeft["Y"],
              let xRightTopRatio = dicTopRight["X"],
              let yRightTopRatio = dicTopRight["Y"],
              let xBottomRightRatio = dicBottomRight["X"],
              let yBottomRightRatio = dicBottomRight["Y"],
              let xLeftBottomRatio = dicBottomLeft["X"],
              let yLeftBottomRatio = dicBottomLeft["Y"] else {
            return CGRect.zero
        }

        // 由于截图只能矩形，所以截图不规则四边形的最大外围
        let xMinLeft = CGFloat(min(xLeftTopRatio, xLeftBottomRatio))
        let xMaxRight = CGFloat(max(xRightTopRatio, xBottomRightRatio))

        let yMinTop = CGFloat(min(yLeftTopRatio, yRightTopRatio))
        let yMaxBottom = CGFloat(max(yLeftBottomRatio, yBottomRightRatio))

        let imgW = srcCodeImage.size.width
        let imgH = srcCodeImage.size.height

        // 宽高反过来计算
        let rect = CGRect(x: xMinLeft * imgH, y: yMinTop * imgW, width: (xMaxRight - xMinLeft) * imgH, height: (yMaxBottom - yMinTop) * imgW)
        return rect
    }

    // MARK: - ---图像处理

    /**
     @brief  图像中间加logo图片
     @param srcImg    原图像
     @param LogoImage logo图像
     @param logoSize  logo图像尺寸
     @return 加Logo的图像
     */
    public static func addImageLogo(srcImg: UIImage, logoImg: UIImage, logoSize: CGSize) -> UIImage {
        UIGraphicsBeginImageContext(srcImg.size)
        srcImg.draw(in: CGRect(x: 0, y: 0, width: srcImg.size.width, height: srcImg.size.height))
        let rect = CGRect(x: srcImg.size.width / 2 - logoSize.width / 2, y: srcImg.size.height / 2 - logoSize.height / 2, width: logoSize.width, height: logoSize.height)
        logoImg.draw(in: rect)
        let resultingImage = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        return resultingImage!
    }

    // 图像缩放
    static func resizeImage(image: UIImage, quality: CGInterpolationQuality, rate: CGFloat) -> UIImage? {
        var resized: UIImage?
        let width = image.size.width * rate
        let height = image.size.height * rate

        UIGraphicsBeginImageContext(CGSize(width: width, height: height))
        guard let context = UIGraphicsGetCurrentContext() else {
            UIGraphicsEndImageContext()
            return nil
        }
        context.interpolationQuality = quality
        image.draw(in: CGRect(x: 0, y: 0, width: width, height: height))

        resized = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()

        return resized
    }

    // 图像裁剪
    static func imageByCroppingWithStyle(srcImg: UIImage, rect: CGRect) -> UIImage? {
        guard let imageRef = srcImg.cgImage,
              let imagePartRef = imageRef.cropping(to: rect) else {
            return nil
        }
        let cropImage = UIImage(cgImage: imagePartRef)

        return cropImage
    }

    // 图像旋转
    static func imageRotation(image: UIImage, orientation: UIImage.Orientation) -> UIImage {
        var rotate: Double = 0.0
        var rect: CGRect
        var translateX: CGFloat = 0.0
        var translateY: CGFloat = 0.0
        var scaleX: CGFloat = 1.0
        var scaleY: CGFloat = 1.0

        switch orientation {
        case .left:
            rotate = .pi / 2
            rect = CGRect(x: 0, y: 0, width: image.size.height, height: image.size.width)
            translateX = 0
            translateY = -rect.size.width
            scaleY = rect.size.width / rect.size.height
            scaleX = rect.size.height / rect.size.width
        case .right:
            rotate = 3 * .pi / 2
            rect = CGRect(x: 0, y: 0, width: image.size.height, height: image.size.width)
            translateX = -rect.size.height
            translateY = 0
            scaleY = rect.size.width / rect.size.height
            scaleX = rect.size.height / rect.size.width
        case .down:
            rotate = .pi
            rect = CGRect(x: 0, y: 0, width: image.size.width, height: image.size.height)
            translateX = -rect.size.width
            translateY = -rect.size.height
        default:
            rotate = 0.0
            rect = CGRect(x: 0, y: 0, width: image.size.width, height: image.size.height)
            translateX = 0
            translateY = 0
        }

        UIGraphicsBeginImageContext(rect.size)
        guard let context = UIGraphicsGetCurrentContext(),
              let cgImage = image.cgImage else {
            UIGraphicsEndImageContext()
            return image
        }

        // 做CTM变换
        context.translateBy(x: 0.0, y: rect.size.height)
        context.scaleBy(x: 1.0, y: -1.0)
        context.rotate(by: CGFloat(rotate))
        context.translateBy(x: translateX, y: translateY)

        context.scaleBy(x: scaleX, y: scaleY)
        // 绘制图片
        context.draw(cgImage, in: CGRect(x: 0, y: 0, width: rect.size.width, height: rect.size.height))
        let newPic = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()

        return newPic ?? image
    }

    deinit {
        //        print("LBXScanWrapper deinit")
    }
}
