//
//  YHSearchInfoBar.swift
//  galaxy
//
//  Created by edy on 2024/4/8.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHSearchInfoBar: UIView {

    static let height = 36.0
    static let maxWordsCount = 50
    
    var textChange: ((String?) -> Void)?
    var searchBlock: ((String?) -> Void)?
    var placeHolder: String? {
        didSet {
            if let placeholder = placeHolder {
                textField.attributedPlaceholder = NSAttributedString(string: placeholder, attributes: [NSAttributedString.Key.foregroundColor: UIColor.placeHolderColor])
            }
        }
    }
    private lazy var contentView = {
        let view = UIView()
        view.backgroundColor = UIColor(hex: 0xF4F6FA)
        view.clipsToBounds = true
        return view
    }()
    
    lazy var searchImgView: UIImageView = {
        let imgView = UIImageView(image: UIImage(named: "home_search_icon"))
        return imgView
    }()
    
    lazy var textField: UITextField = {
        let textField = UITextField()
        textField.backgroundColor = .clear
        textField.attributedPlaceholder = NSAttributedString(string: "香港身份".local, attributes: [NSAttributedString.Key.foregroundColor: UIColor.yhGreyColor])
        textField.font = UIFont.PFSC_M(ofSize: 12)
        textField.tintColor = UIColor.brandMainColor
        textField.textColor = UIColor.mainTextColor
        textField.clearButtonMode = .whileEditing
        textField.addTarget(self, action: #selector(textFieldChanged(textField:)), for: .editingChanged)
        textField.delegate = self
        return textField
    }()
    
    lazy var searchBtn: UIButton = {
        let btn = UIButton()
        btn.setTitle("搜索".local, for: .normal)
        btn.titleLabel?.font = UIFont.PFSC_R(ofSize: 12)
        btn.setTitleColor(.white, for: .normal)
        btn.backgroundColor = UIColor.brandMainColor
        btn.addTarget(self, action: #selector(searchBtnClicked), for: .touchUpInside)
        return btn
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        self.createUI()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    @objc func searchBtnClicked() {
        textField.resignFirstResponder()
        if let searchBlock = searchBlock {
            searchBlock(textField.text)
        }
    }
    
    func setSearchButtonEnable(_ enable: Bool) {
        searchBtn.isUserInteractionEnabled = enable
        searchBtn.alpha = (enable ? 1.0 : 0.5)
    }
    
    func createUI() {
        
        self.backgroundColor = .white
        setSearchButtonEnable(true)

        contentView.addSubview(searchImgView)
        contentView.addSubview(textField)
        contentView.addSubview(searchBtn)
        self.addSubview(contentView)
        
        contentView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        
        searchImgView.snp.makeConstraints { make in
            make.width.height.equalTo(16.0)
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(12.0)
        }
        
        textField.snp.makeConstraints { make in
            make.left.equalTo(searchImgView.snp.right).offset(8)
            make.right.equalTo(searchBtn.snp.left).offset(-4)
            make.top.bottom.equalToSuperview()
        }
        
        searchBtn.snp.makeConstraints { make in
            make.size.equalTo(CGSize(width: 56, height: Self.height-8.0))
            make.centerY.equalToSuperview()
            make.right.equalToSuperview().offset(-4)
        }
    }
    
    @objc func textFieldChanged(textField: UITextField) {
        if let textChange = textChange {
            textChange(textField.text)
        }
    }
}

extension YHSearchInfoBar: UITextFieldDelegate {
    
    func textField(_ textField: UITextField, shouldChangeCharactersIn range: NSRange, replacementString string: String) -> Bool {
        
        if string.isEmpty == true { // 点击删除
            return true
        }
        
        let newText = (textField.text! as NSString).replacingCharacters(in: range, with: string)
        if newText.count > Self.maxWordsCount {
            YHHUD.flash(message: "搜索限制最多\(Self.maxWordsCount)个字符")
            return false
        }
        return true
    }
}
