//
//  YHVisaRenewalWorkflowCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/10/10.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHVisaRenewalWorkflowCell: UITableViewCell {

    static let cellReuseIdentifier = "YHVisaRenewalWorkflowCell"
    private let circleWidth: CGFloat = 8
    
    private var stepLabels: [UILabel] = []
    private var circleViews: [UIView] = []
    private var dashLines: [YHDashLineView] = []
    
    private lazy var iconView: UIView = {
        let view = UIView()
        view.backgroundColor = .brandMainColor
        return view
    }()
    
    private lazy var infoTitleLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_B(ofSize: 14)
        label.textColor = .mainTextColor
        return label
    }()
    
    private lazy var subContainerView: YHBaseDynamicCornerRadiusView = {
        let view = YHBaseDynamicCornerRadiusView(cornerRadius: 6, corner: .allCorners)
        view.backgroundColor = .contentBkgColor
        return view
    }()
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupCellInfo(title: String, workflows: [String]) {
        infoTitleLabel.text = title
        stepLabels.forEach {
            $0.removeFromSuperview()
        }
        circleViews.forEach {
            $0.removeFromSuperview()
        }
        dashLines.forEach {
            $0.removeFromSuperview()
        }
        stepLabels.removeAll()
        circleViews.removeAll()
        dashLines.removeAll()
        let count = workflows.count
        workflows.enumerated().forEach { [weak self] index, title in
            guard let self = self else {
                return
            }
            let circleView = self.getCircleView()
            let label = self.getStepLabel(title)
            self.stepLabels.append(label)
            self.circleViews.append(circleView)
            self.subContainerView.addSubview(label)
            self.subContainerView.addSubview(circleView)
            
            label.snp.makeConstraints { make in
                if index == 0 {
                    make.top.equalTo(self.infoTitleLabel.snp.bottom).offset(12)
                } else {
                    make.top.equalTo(self.stepLabels[index - 1].snp.bottom).offset(16)
                }
                make.left.equalTo(circleView.snp.right).offset(8)
                make.right.lessThanOrEqualToSuperview().offset(-16)
                if index == count - 1 {
                    make.bottom.equalToSuperview().offset(-20)
                }
                
            }
            circleView.snp.makeConstraints { make in
                make.left.equalToSuperview().offset(16)
                make.top.equalTo(label.snp.top).offset(6)
                make.width.height.equalTo(self.circleWidth)
            }
            if index != count - 1 {
                let dashLine = self.getDashLineView()
                self.dashLines.append(dashLine)
                self.subContainerView.addSubview(dashLine)
            }
        }
        guard dashLines.count > 0 else {
            return
        }
        dashLines.enumerated().forEach { [weak self] index, line in
            guard let self = self else {
                return
            }
            guard self.circleViews.count > index + 1 else {
                return
            }
            line.snp.makeConstraints { make in
                make.width.equalTo(1)
                make.centerX.equalTo(self.circleViews[index])
                make.top.equalTo(self.circleViews[index].snp.bottom).offset(1)
                make.bottom.equalTo(self.circleViews[index + 1].snp.top).offset(-1)
            }
        }
    }

}

extension YHVisaRenewalWorkflowCell {
    private func setupUI() {
        selectionStyle = .none
        backgroundColor = .clear
        contentView.addSubview(subContainerView)
        subContainerView.addSubview(iconView)
        subContainerView.addSubview(infoTitleLabel)
        subContainerView.setContentHuggingPriority(.defaultLow, for: .vertical)
        subContainerView.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(20)
            make.left.equalToSuperview().offset(20)
            make.right.equalToSuperview().offset(-20)
            make.bottom.equalToSuperview()
        }
        
        iconView.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(18)
            make.left.equalToSuperview()
            make.width.equalTo(4)
            make.height.equalTo(14)
        }
        
        infoTitleLabel.snp.makeConstraints { make in
            make.centerY.equalTo(iconView)
            make.left.equalTo(iconView.snp.right).offset(12)
            make.right.lessThanOrEqualToSuperview().offset(-16)
            make.bottom.lessThanOrEqualToSuperview().offset(-20)
        }
    }
    
    private func getStepLabel(_ title: String) -> UILabel {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 13)
        label.textColor = .mainTextColor70
        label.numberOfLines = 0
        label.lineBreakMode = .byCharWrapping
        let paragraphStyle = NSMutableParagraphStyle()
        paragraphStyle.lineHeightMultiple = 1.1
        let attributedText = NSAttributedString(
            string: title,
            attributes: [NSAttributedString.Key.paragraphStyle: paragraphStyle]
        )
        label.attributedText = attributedText
        return label
    }
    
    private func getDashLineView() -> YHDashLineView {
        let view = YHDashLineView(strokeColor: UIColor.brandMainColor30.cgColor, lineWidth: 1, lineDashPattern: [2, 2], isHorizontal: false)
        return view
    }
    
    private func getCircleView() -> UIView {
        let view = UIView()
        view.layer.cornerRadius = circleWidth / 2.0
        view.clipsToBounds = true
        view.layer.borderColor = UIColor.brandMainColor30.cgColor
        view.layer.borderWidth = 1.5
        return view
    }
}
