//
//  YHWorkingHoursStatementCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/13.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import AttributedString
import UIKit

class YHWorkingHoursStatementCell: YHResignDocumentCell {
    static let cellReuseIdentifier = "YHWorkingHoursStatementCell"
    private let maxCount = 1000
    var textViewDidChanged: ((UITextView) -> Void)?
    var textDidExceedsMaxLimit: ((Int) -> Void)?
    
    private lazy var infoDetailLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 13)
        label.textColor = UIColor(hexString: "#8993A2")
        label.numberOfLines = 0
        label.lineBreakMode = .byCharWrapping
        return label
    }()
    
    private lazy var remarkTextViewContainer: YHBaseDynamicCornerRadiusView = {
        let view = YHBaseDynamicCornerRadiusView(cornerRadius: 1, corner: .allCorners)
        view.backgroundColor = .contentBkgColor
        return view
    }()
    
    private lazy var remarkTextView: UITextView = {
        let view = UITextView()
        view.backgroundColor = .clear
        view.font = .PFSC_R(ofSize: 14)
        view.textColor = .mainTextColor
        view.tintColor = .mainTextColor
        view.returnKeyType = .default
        view.enablesReturnKeyAutomatically = true
        view.delegate = self
        view.textContainerInset = .zero
        view.textContainer.lineFragmentPadding = 0
        return view
    }()
    
    private lazy var editTipsLabel: UILabel = {
        let label = UILabel()
        label.isHidden = true
        label.font = .PFSC_R(ofSize: 14)
        label.textColor = UIColor.init(hexString: "#B9C1CC")
        return label
    }()
    
    private lazy var countLabel: UILabel = {
        let label = UILabel()
        label.text = "0/\(maxCount)"
        label.font = .PFSC_R(ofSize: 14)
        label.textColor = UIColor(hexString: "#D5DAE1")
        return label
    }()
    
    private lazy var bottomLineView: UIView = {
        let view = UIView()
        view.backgroundColor = UIColor(hexString: "#E9ECF0")
        return view
    }()
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupCellInfo(placeholder: String, remark: String?, detail: ASAttributedString, isShowBottomLine: Bool) {
        editTipsLabel.text = placeholder
        editTipsLabel.isHidden = remark?.isEmpty == false
        remarkTextView.text = remark
        infoDetailLabel.attributed.text = detail
        bottomLineView.isHidden = !isShowBottomLine
    }
}

extension YHWorkingHoursStatementCell {
    private func setupUI() {
        updateCellCorner(.single)
        subContainerView.addSubview(infoDetailLabel)
        subContainerView.addSubview(remarkTextViewContainer)
        subContainerView.addSubview(bottomLineView)
        remarkTextViewContainer.addSubview(editTipsLabel)
        remarkTextViewContainer.addSubview(remarkTextView)
        remarkTextViewContainer.addSubview(countLabel)
        
        infoDetailLabel.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(16)
            make.left.equalToSuperview().offset(18)
            make.right.lessThanOrEqualToSuperview().offset(-18)
        }
        
        remarkTextViewContainer.snp.makeConstraints { make in
            make.top.equalTo(infoDetailLabel.snp.bottom).offset(8)
            make.left.equalTo(infoDetailLabel.snp.left)
            make.right.equalToSuperview().offset(-18)
            make.height.equalTo(160)
            make.bottom.equalToSuperview().offset(-16)
        }
        
        editTipsLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(12)
            make.top.equalToSuperview().offset(12)
            make.right.lessThanOrEqualToSuperview().offset(-12)
        }
        
        countLabel.snp.makeConstraints { make in
            make.right.equalToSuperview().offset(-12)
            make.bottom.equalToSuperview().offset(-12)
            make.left.greaterThanOrEqualToSuperview().offset(12)
        }
        
        remarkTextView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(12)
            make.top.equalToSuperview().offset(12)
            make.right.equalToSuperview().offset(-12)
            make.bottom.equalTo(countLabel.snp.top)
        }
        
        bottomLineView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(18)
            make.right.equalToSuperview().offset(-18)
            make.height.equalTo(0.5)
            make.bottom.equalToSuperview()
        }
    }
}

extension YHWorkingHoursStatementCell: UITextViewDelegate {
    func textViewDidChange(_ textView: UITextView) {
        editTipsLabel.isHidden = textView.text.count > 0
        let textCount = textView.text?.utf16.count ?? 0
        countLabel.text = "\(textCount)/\(maxCount)"
        textViewDidChanged?(textView)
    }
    
    func textView(_ textView: UITextView, shouldChangeTextIn range: NSRange, replacementText text: String) -> Bool {
        if textView == remarkTextView {
            /// 正在输入拼音
            if let textRange = textView.markedTextRange, range.length == 0 {
                /// 刚输入拼音的时候总字数
                let start = textView.offset(from: textView.beginningOfDocument, to: textRange.start)
                return start <= maxCount
            }
            let count = (textView.text?.utf16.count ?? 0) - range.length + text.utf16.count
            if count > maxCount && text == "" {
                return true
            }
            guard count > maxCount else {
                return true
            }
            textDidExceedsMaxLimit?(maxCount)
            return false
        }
        return true
    }
}
