//
//  YHInformationFillVC.swift
//  galaxy
//
//  Created by edy on 2024/4/11.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import SafariServices

enum YHOrderStage: Int {
    case apply = 1  // 申请阶段
    case certificate = 2 // 办证阶段
    case resign = 3 // 续签
    case permanentResidence = 4 // 永居
}

enum YHRedirectPage: String {
    
    // 通用
    case serviceTab = "service_tab" // 服务tab
    case lifeTab = "life_tab" // 首页-生活tab
    case newsTab = "news_tab" // 首页-资讯tab
    
    // stage = 1 (申请阶段) 可以进入的路由页面
    case myInfo = "my_info"  // 我的信息
    case myCertificate = "my_certificate" // 我的证件
    case myScore = "my_score" // 我的评分
    case myPlan = "my_plan" // 我的方案
    case myDocument = "my_document" // 我的文书
    case mySignature = "my_signature" // 我的签字
    
    // stage = 2 (办证阶段) 可以进入的路由页面
    case approveResult = "approve_result" // 港府获批结果
    case familyInfoConfirm = "family_info_confirm" // 家庭信息确认
    case addDependent = "add_dependent" // 新增受养人
    case appointForIdCard = "appoint_for_id_card" // 赴港办证预约
    case payElectronicVisa = "pay_electronic_visa" // 电子签证缴费
    case travelGuide = "travel_guide" // 赴港行程指引
    case prepareTravelCertificate = "prepare_travel_certificate" // 旅行证件准备
    case uploadPassCertificate = "upload_pass_certificate" // 上传过关证件
    case principalApprovalSuccess = "principal_approval_success" // 原则批获批成功
    
    // stage = 3 (续签阶段) 可以进入的路由页面  续签阶段可以进入 办证阶段能进入的路由界面
    case renewalGuide = "renewal_guide" // 身份续签指引
    case renewalPlan = "renewal_plan" // 我的续签方案
    case stayInHKRecord = "stay_in_hk_record" // 添加在港记录
    case renewalCertificate = "renewal_certificate" // 续签证件管理
    case renewalMaterial = "renewal_material" // 续签资料管理
    case renewalDocument = "renewal_document" // 续签文书管理
    
    case renewalInfoConfirm = "renewal_info_confirm" // 续签信息确认
    case renewalSubmitInHkAppointment = "renewal_submit_in_hk_appointment" // 在港递交预约
    case renewalResult = "renewal_result" // 查看续签结果
    case renewalSupplement = "renewal_supplement" // 续签补件
    case renewalVisaPay = "renewal_visa_pay" // 在港签证缴费
    case renewalTravelCertificateUpload = "renewal_travel_certificate_upload" // 上传旅行证件
}

class YHMessageDetailListVC: YHBaseViewController {
    
    var type: Int = -1
    var msgArr: [YHDetailMessageModel] = []
    // 滑动过程中出现在屏幕上的未读消息集合
    var visibleUnreadMsgIdSet = Set<String>()
    var visibleUnreadMsgArr = [YHDetailMessageModel]()
    var currentPage = 1
    var isLastPage = false
    
    lazy var viewModel = {
        let model = YHMsgViewModel()
        return model
    }()
    
    let principleViewModel: YHPrincleViewModel = YHPrincleViewModel()
    let serviceCenterViewModel = YHServiceCenterViewModel()
    
    func getRoutersForOrderStage(_ stage: Int) -> [YHRedirectPage] {
        
        if stage == YHOrderStage.apply.rawValue {
            return [.myInfo,
                    .myCertificate,
                    .myScore,
                    .myPlan,
                    .myDocument,
                    .mySignature ]
        }
        
        if stage == YHOrderStage.certificate.rawValue {
            
            return [.approveResult,
                    .familyInfoConfirm,
                    .addDependent,
                    .appointForIdCard,
                    .payElectronicVisa,
                    .travelGuide,
                    .prepareTravelCertificate,
                    .uploadPassCertificate,
                    .principalApprovalSuccess ]
        }
        
        if stage == YHOrderStage.resign.rawValue ||
           stage == YHOrderStage.permanentResidence.rawValue {
            
            return [.approveResult,
                    .familyInfoConfirm,
                    .addDependent,
                    .appointForIdCard,
                    .payElectronicVisa,
                    .travelGuide,
                    .prepareTravelCertificate,
                    .uploadPassCertificate,
                    .principalApprovalSuccess,
                    
                    .renewalGuide,
                    .renewalPlan,
                    .stayInHKRecord,
                    .renewalCertificate,
                    .renewalMaterial,
                    .renewalDocument,
                    .renewalInfoConfirm,
                    .renewalSubmitInHkAppointment,
                    .renewalResult,
                    .renewalSupplement,
                    .renewalVisaPay,
                    .renewalTravelCertificateUpload]
            
        }
        return []
    }
    
    lazy var navBar: YHCustomNavigationBar = {
        let bar = YHCustomNavigationBar.navBar()
        bar.backgroundColor = .white
        if let title = self.title, !title.isEmpty {
            bar.title = title
        }
        bar.showRightButtonType(.clean)
        bar.backBlock = {
            [weak self] in
            guard let self = self else { return }
            self.navigationController?.popViewController(animated: true)
        }
        bar.rightBtnClick = {
            [weak self] in
            guard let self = self else { return }
            let unreadMessages = self.msgArr.filter { !$0.isRead }
            if unreadMessages.count <= 0 {
                YHHUD.flash(message: "暂无未读消息~")
                return
            }
            YHCommonAlertView.show("清除未读", "确定要清除所有未读提示吗？", "取消", "确认") {
                self.markAllMsgsRead()
            }
        }
        return bar
    }()
    
    lazy var tableView: UITableView = {
        let tableView = UITableView(frame: .zero, style: .grouped)
        tableView.estimatedSectionHeaderHeight = 1.0
        tableView.estimatedSectionFooterHeight = 1.0
        tableView.showsVerticalScrollIndicator = false
        tableView.separatorStyle = .none
        tableView.delegate = self
        tableView.dataSource = self
        tableView.backgroundColor = UIColor(hex: 0xF8F8F8)
        tableView.backgroundView = emptyDataTipsView
        tableView.register(UITableViewCell.self, forCellReuseIdentifier: "UITableViewCell")
        tableView.register(YHDetailMessageCell.self, forCellReuseIdentifier: YHDetailMessageCell.cellReuseIdentifier)
        tableView.es.addInfiniteScrolling {
            self.getMsgList(more: true)
        }
        return tableView
    }()
    
    lazy var emptyDataTipsView: YHEmptyDataView = {
        let view = YHEmptyDataView.createView("暂无消息", kEmptyMessageBgName)
        let topMargin = 222.0
        view.topMargin = topMargin
        view.isHidden = true
        return view
    }()

    override func viewDidLoad() {
        super.viewDidLoad()
        setupUI()
        getMsgList()
    }
    
    func setupUI() {
        
        gk_navigationBar.isHidden = true
        view.backgroundColor = .white
        view.addSubview(tableView)
        view.addSubview(navBar)
        navBar.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.top.equalToSuperview().offset(k_Height_StatusBar)
            make.height.equalTo(k_Height_NavContentBar)
        }
        tableView.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.top.equalToSuperview().offset(k_Height_NavigationtBarAndStatuBar)
            make.bottom.equalToSuperview()
        }
    }
    
    func getMsgList(more: Bool = false) {
        
        if isLastPage {
            tableView.es.stopLoadingMore()
            return
        }
        
        if more {
            currentPage += 1
        } else {
            currentPage = 1
        }
        
        self.viewModel.getMsgDetailList(type: self.type, page: currentPage) {
            [weak self] _, isLastPage in
            guard let self = self else { return }
            msgArr.removeAll()
            msgArr.append(contentsOf: self.viewModel.msgDetailList)
            emptyDataTipsView.isHidden = msgArr.count > 0
            self.isLastPage = isLastPage
            tableView.es.stopLoadingMore()
            if isLastPage {
                tableView.es.noticeNoMoreData()
            }
            self.tableView.reloadData {
                [weak self] in
                guard let self = self else { return }
                self.addUpAppearUnreadMessages()
                self.markAppearMessagsReaded()
            }
        }
    }
    
    func markAllMsgsRead() {
        YHHUD.show(.progress(message: "清除中..."))
        self.viewModel.readMessages(type: self.type, msgIds: nil, isAllRead: true) {
            [weak self] success, error in
            YHHUD.hide()
            guard let self = self else { return }
            if success {
                YHHUD.flash(message: "清除成功")
                for item in self.msgArr {
                    item.isRead = true
                }
                self.tableView.reloadData()
                return
            }
            var msg = "清除失败"
            if let error = error {
                if error.errorMsg.count > 0 {
                    msg = error.errorMsg
                }
            }
            YHHUD.flash(message: msg)
        }
    }
    
    func markAppearMessagsReaded() {
        var msgIds: [Int] = []
        var resultArr: [YHDetailMessageModel] = []
        for msgItem in visibleUnreadMsgArr {
            if let msgId = Int(msgItem.id), msgItem.isRead == false {
                msgIds.append(msgId)
                resultArr.append(msgItem)
            }
        }
        if msgIds.count <= 0 { return }
        self.viewModel.readMessages(type: self.type, msgIds: msgIds, isAllRead: false) {
            [weak self] success, _ in
            guard let self = self else { return }
            if success {
                for msgItem in resultArr {
                    msgItem.isRead = true
                }
                self.tableView.reloadData()
                return
            }
        }
    }
    
    static func detailWithTripUrl(_ itemUrl: String) {
        if !itemUrl.isEmpty, let token = YHLoginManager.shared.userModel?.token, token.count > 5 {
            var url = itemUrl + "&param=\(YHLoginManager.shared.h5Token)"
            url = url + "&app_platform=ios" + "&app_version=" + UIDevice.appVersion()
            printLog("UURL: \(url)")
            let vc = YHH5WebViewVC()
            vc.url = url
            vc.isFullScreenFlag = true
            vc.isNeedRefreshDataFlag = true
            UIViewController.current?.navigationController?.pushViewController(vc)
        }
    }
}

extension YHMessageDetailListVC: UITableViewDelegate, UITableViewDataSource {
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return msgArr.count
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        
        guard let cell = tableView.dequeueReusableCell(withIdentifier: YHDetailMessageCell.cellReuseIdentifier, for: indexPath) as? YHDetailMessageCell else { return UITableViewCell() }
        if 0 <= indexPath.row && indexPath.row < msgArr.count {
            let item = msgArr[indexPath.row]
            cell.updateModel(item, type: self.type)
        }
        return cell
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return UITableView.automaticDimension
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        
        if 0 <= indexPath.row && indexPath.row < msgArr.count {
            let item = msgArr[indexPath.row]
            printLog("order_id: \(item.orderId)")
            
            if !item.isRead, let msgId = Int(item.id), msgId > 0 { // 消息为未读消息
                self.viewModel.readMessages(type: self.type, msgIds: [msgId], isAllRead: false) { 
                    [weak self] success, _ in
                    guard let self = self else { return }
                    if success {
                        item.isRead = true
                        self.tableView.reloadData()
                    }
                }
            }
            
            if self.type == YHMessageType.business.rawValue {
                return
            }
            
            if self.type == YHMessageType.trip.rawValue {
                Self.detailWithTripUrl(item.url)
                return
            }
            
            printLog("redirectPage:\(item.redirectPage)")

            if item.redirectPage == YHRedirectPage.serviceTab.rawValue { // 服务tab页
                UIViewController.current?.navigationController?.popToRootViewController(animated: false)
                goTabBarBy(tabType: .service)
                return
                
            }
            
            if item.redirectPage == YHRedirectPage.lifeTab.rawValue { // 首页-生活tab
                UIViewController.current?.navigationController?.popToRootViewController(animated: false)
                goTabBarBy(tabType: .home)
                NotificationCenter.default.post(name: YhConstant.YhNotification.didJumpToHomePageLifeTabNotification, object: nil)
                return
                
            }
            
            if item.redirectPage == YHRedirectPage.newsTab.rawValue { // 首页-资讯tab
                UIViewController.current?.navigationController?.popToRootViewController(animated: false)
                goTabBarBy(tabType: .home)
                NotificationCenter.default.post(name: YhConstant.YhNotification.didJumpToHomePageNewsTabNotification, object: nil)
                return
                
            }
                
            // 处理订单相关逻辑
            if let orderId = Int(item.orderId) {
                printLog("orderId: \(orderId)")
                self.requestOrderStatus(orderId: orderId) {
                    status in
                    // status 1 申请阶段 2办证阶段 3续期阶段
                    self.handleRouter(message: item, stage: status) {
                        // 没有满足相应跳转条件默认跳转到服务页
                        UIViewController.current?.navigationController?.popToRootViewController(animated: false)
                        goTabBarBy(tabType: .service)
                    }
                }
            }
        }
    }

    private func tableView(_ tableView: UITableView, viewForHeaderInSection section: Int) -> CGFloat {
        return 1.0
    }
    
    private func tableView(_ tableView: UITableView, viewForFooterInSection section: Int) -> CGFloat {
        return 1.0
    }
    
    func tableView(_ tableView: UITableView, viewForFooterInSection section: Int) -> UIView? {
        
        return UIView()
    }
    
    func tableView(_ tableView: UITableView, viewForHeaderInSection section: Int) -> UIView? {

        return UIView()
    }
    
    func scrollViewDidEndDragging(_ scrollView: UIScrollView, willDecelerate decelerate: Bool) {
        if scrollView == tableView && !decelerate {
            // 在这里处理滑动停止事件
            print("TableView 滑动停止")
            markAppearMessagsReaded()
        }
    }
    
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        addUpAppearUnreadMessages()
    }
    
    // 累计出现在屏幕上的未读消息集合
    func addUpAppearUnreadMessages() {
        let visibleCells = tableView.visibleCells as? [YHDetailMessageCell] ?? []
        if visibleCells.count <= 0 { return }
        for cell in visibleCells {
            if let msgItem = cell.model {
                if !msgItem.isRead {
                    if !visibleUnreadMsgIdSet.contains(msgItem.id) {
                        visibleUnreadMsgIdSet.insert(msgItem.id)
                        visibleUnreadMsgArr.append(msgItem)
                    }
                }
            }
        }
        printLog("visiblemsgIdSet:/n \(visibleUnreadMsgIdSet)")
    }
    
    func handleRouter(message: YHDetailMessageModel, stage: Int, defaultRouterBlock: (() -> Void)?) {
            // status 1 申请阶段 2办证阶段 3续期阶段
           if stage == YHOrderStage.apply.rawValue { // 申请阶段
                if self.type == YHMessageType.infoSubmit.rawValue { // 资料递交
                    if !message.url.isEmpty {
                        if let url = URL(string: message.url) {
                            if  message.url.contains("signatureQrcode/") {
                                // 特殊处理 签字确认
                                let vc = YHH5WebViewVC()
                                vc.url = message.url
                                vc.isNeedRefreshDataFlag = true
                                UIViewController.current?.navigationController?.pushViewController(vc)
                            } else {
                                let safariViewController = SFSafariViewController(url: url)
                                safariViewController.dismissButtonStyle = .close
                                safariViewController.modalPresentationStyle = .fullScreen
                                present(safariViewController, animated: true, completion: nil)
                            }
                        } else {
                            YHHUD.flash(message: "url无效")
                        }
                        return
                    }
                }
            }
            
            // 根据订单状态获取其可以进入的路由数组
            let routers = self.getRoutersForOrderStage(stage)
            let routerStringArr = routers.map { page in
                return page.rawValue
            }
            printLog("\(message.redirectPage)")
            if routerStringArr.contains(message.redirectPage) {
                self.routeToPage(message)
                
            } else {
                defaultRouterBlock?()
            }
    }
    
    func routeToPage(_ msg: YHDetailMessageModel) {
        
        if msg.redirectPage == YHRedirectPage.myInfo.rawValue { // 我的信息
            if let orderId = Int(msg.orderId) {
                let vc = YHInformationPerfectListVC()
                vc.orderId = orderId
                navigationController?.pushViewController(vc)
            }
            
        } else if msg.redirectPage == YHRedirectPage.myCertificate.rawValue { // 我的证件
            if let orderId = Int(msg.orderId) {
                let vc = YHMineCertificateEntryViewController()
                vc.orderId = orderId
                navigationController?.pushViewController(vc)
            }
            
        } else if msg.redirectPage == YHRedirectPage.myScore.rawValue { // 我的评分
            if let orderId = Int(msg.orderId) {
                let vc = YHScoreDetailViewController()
                vc.orderId = orderId
                navigationController?.pushViewController(vc)
            }
        
        } else if msg.redirectPage == YHRedirectPage.myPlan.rawValue { // 我的方案
            YHMyNewSchemeViewController.shared.goToSchemePage()

        } else if msg.redirectPage == YHRedirectPage.myDocument.rawValue { // 我的文书
            if let orderId = Int(msg.orderId) {
                let vc = YHMyDocumentsListViewController()
                vc.orderId = orderId
                navigationController?.pushViewController(vc)
            }
            
        } else if msg.redirectPage == YHRedirectPage.mySignature.rawValue { // 我的签字
            if let orderId = Int(msg.orderId) {
                let vc = YHMySignatureListViewController()
                vc.orderId = orderId
                navigationController?.pushViewController(vc)
            }
            
        } else if msg.redirectPage == YHRedirectPage.approveResult.rawValue { // 港府获批结果(正式批)
                let vc = YHOfficialApprovalResultVC()
                navigationController?.pushViewController(vc)
            
            } else if msg.redirectPage == YHRedirectPage.familyInfoConfirm.rawValue { // 家庭信息确认
                if let orderId = Int(msg.orderId) {
                    let vc = YHFamilyInfoConfirmViewController()
                    vc.orderId = orderId
                    navigationController?.pushViewController(vc)
                }
            
            } else if msg.redirectPage == YHRedirectPage.addDependent.rawValue { // 新增受养人
                if let orderId = Int(msg.orderId) {
                    let vc = YHAddAdoptersViewController()
                    vc.orderId = orderId
                    navigationController?.pushViewController(vc)
                }
                
            } else if msg.redirectPage == YHRedirectPage.appointForIdCard.rawValue { // 赴港办证预约
                if let orderId = Int(msg.orderId) {
                    let vc = YHCertificateAppointViewController()
                    vc.orderId = orderId
                    navigationController?.pushViewController(vc)
                }
                
            } else if msg.redirectPage == YHRedirectPage.payElectronicVisa.rawValue { // 电子签证缴费
                if let orderId = Int(msg.orderId) {
                    let vc = YHVisaPaymentVC()
                    vc.orderID = orderId
                    navigationController?.pushViewController(vc)
                }
                
            } else if msg.redirectPage == YHRedirectPage.travelGuide.rawValue { // 赴港行程指引
                if let orderId = Int(msg.orderId), let token = YHLoginManager.shared.userModel?.token, token.count > 5 {
                    var url = YHBaseUrlManager.shared.curH5URL() + "superAppBridge.html#/tohkList"
                    url = url + "?paramO=\(orderId)" + "&param=" + YHLoginManager.shared.h5Token
                    url = url + "&app_platform=ios" + "&app_version=" + UIDevice.appVersion()
                    printLog("UURL: \(url)")
                    let vc = YHH5WebViewVC()
                    vc.url = url
                    vc.isFullScreenFlag = false
                    vc.navTitle = "赴港行程指引"
                    vc.isNeedRefreshDataFlag = true
                    UIViewController.current?.navigationController?.pushViewController(vc)
                }
                
            } else if msg.redirectPage == YHRedirectPage.prepareTravelCertificate.rawValue { // 旅行证件准备
                if let orderId = Int(msg.orderId) {
                    let vc = YHTravelDocsPreparationListViewController()
                    vc.orderId = orderId
                    navigationController?.pushViewController(vc)
                }
            } else if msg.redirectPage == YHRedirectPage.uploadPassCertificate.rawValue { // 上传过关证件
                if let orderId = Int(msg.orderId) {
                    let vc = YHUploadCertificateVC()
                    vc.orderId = orderId
                    navigationController?.pushViewController(vc)
                }
            } else if msg.redirectPage == YHRedirectPage.principalApprovalSuccess.rawValue { // 原则批获批成功
                if let orderId = Int(msg.orderId) {
                    let vc = YHPrincipleApprovedWaitSubmitVC()
                    vc.orderId = orderId
                    navigationController?.pushViewController(vc)
                }
                
           } else if msg.redirectPage == YHRedirectPage.renewalGuide.rawValue { // 身份续签指引 H5
               printLog("身份续签指引")
               if let param = YHLoginManager.shared.userModel?.token, param.count>5 {
                   var url = YHBaseUrlManager.shared.curH5URL() + "superAppBridge.html#/guide/identity"
                   url = url + "?param=" + YHLoginManager.shared.h5Token
                   let vc = YHH5WebViewVC()
                   vc.navTitle = "身份续签指引"
                   vc.url = url
                   vc.isFullScreenFlag = false
                   vc.isNeedRefreshDataFlag = true
                   vc.isSupportWebviewInterBackFlag = true
                   UIViewController.current?.navigationController?.pushViewController(vc)
               }
           } else if msg.redirectPage == YHRedirectPage.renewalPlan.rawValue { // 我的续签方案 H5
               printLog("续签方案")
               if let param = YHLoginManager.shared.userModel?.token, param.count>5, let orderId = Int(msg.orderId) {
                   
                   let stepID = self.serviceCenterViewModel.mainModel.master_id
                   var url = YHBaseUrlManager.shared.curH5URL() + "superAppBridge.html#/report/program"
                   url = url + "?param=" + YHLoginManager.shared.h5Token + "&batchID=\(stepID)" + "&order_id=\(orderId)"
                   let vc = YHH5WebViewVC()
                   vc.navTitle = "我的续签方案"
                   vc.url = url
                   vc.isSupportWebviewInterBackFlag = true
                   UIViewController.current?.navigationController?.pushViewController(vc)
               }

           } else if msg.redirectPage == YHRedirectPage.stayInHKRecord.rawValue { // 添加在港记录
               if let orderId = Int(msg.orderId) {
                   let vc = YHHKImmigrationRecordsVC(orderId: orderId)
                   navigationController?.pushViewController(vc)
               }
           } else if msg.redirectPage == YHRedirectPage.renewalCertificate.rawValue { // 续签证件管理
                 if let orderId = Int(msg.orderId) {
                     let vc = YHResignCertificateListViewController()
                     vc.orderId = orderId
                     vc.batchId = self.serviceCenterViewModel.mainModel.master_id
                     UIViewController.current?.navigationController?.pushViewController(vc)
                 }
                 
           } else if msg.redirectPage == YHRedirectPage.renewalMaterial.rawValue { // 续签资料管理
                 if let orderId = Int(msg.orderId) {
                     let vc = YHResinMaterialManageContainerVC()
                     vc.orderId = orderId
                     vc.batchId = self.serviceCenterViewModel.mainModel.master_id
                     navigationController?.pushViewController(vc)
                 }
             
          } else if msg.redirectPage == YHRedirectPage.renewalDocument.rawValue { // 续签文书管理
              if let orderId = Int(msg.orderId) {
                  let vc = YHResignDocumentManagementVC(orderId: orderId, id: self.serviceCenterViewModel.mainModel.master_id)
                  navigationController?.pushViewController(vc)
              }
          } else if msg.redirectPage == YHRedirectPage.renewalInfoConfirm.rawValue { // 续签信息确认
              if let orderId = Int(msg.orderId) {
                  let vc = YHResignConfirmInfoListViewController()
                  vc.orderId = orderId
                  vc.stepId = self.serviceCenterViewModel.mainModel.master_id
                  navigationController?.pushViewController(vc)
              }
          } else if msg.redirectPage == YHRedirectPage.renewalSubmitInHkAppointment.rawValue { // 在港递交预约
              if let orderId = Int(msg.orderId) {
                  let vc = YHResignAppointContainerViewController()
                  vc.orderId = orderId
                  vc.renewalId = self.serviceCenterViewModel.mainModel.master_id
                  navigationController?.pushViewController(vc)
              }
          } else if msg.redirectPage == YHRedirectPage.renewalResult.rawValue { // 查看续签结果
              if let orderId = Int(msg.orderId) {
                  let vc = YHLookResignResultViewController()
                  vc.orderId = orderId
                  vc.batchId = self.serviceCenterViewModel.mainModel.master_id
                  navigationController?.pushViewController(vc)
              }
          } else if msg.redirectPage == YHRedirectPage.renewalSupplement.rawValue { // 续签补件
              if let orderId = Int(msg.orderId) {
                  let vc = YHResignUploadDocListViewController()
                  vc.orderId = orderId
                  vc.batchId = self.serviceCenterViewModel.mainModel.master_id
                  navigationController?.pushViewController(vc)
              }
          } else if msg.redirectPage == YHRedirectPage.renewalVisaPay.rawValue { // 在港签证缴费
              if let orderId = Int(msg.orderId) {
                  let vc = YHHKVisaRenewalPaymentVC(orderId: orderId, renewalId: self.serviceCenterViewModel.mainModel.master_id)
                  navigationController?.pushViewController(vc)
              }
          } else if msg.redirectPage == YHRedirectPage.renewalTravelCertificateUpload.rawValue { // 上传旅行证件
              if let orderId = Int(msg.orderId) {
                  let vc = YHResignUploadTravelCardListViewController()
                  vc.orderId = orderId
                  vc.batchId = self.serviceCenterViewModel.mainModel.master_id
                  navigationController?.pushViewController(vc)
              }
          }
    }
    
    func requestOrderStatus(orderId: Int, callback: ((_ status: Int) -> Void)?) {
        YHHUD.show(.progress(message: "加载中..."))
        self.serviceCenterViewModel.getServiceMainData(orderId: orderId) {
            [weak self] _, _ in
            YHHUD.hide()
            guard let self = self else { return }
            callback?(self.serviceCenterViewModel.mainModel.stage)
        }
    }
}
