//
//  YHOrderTipsView.swift
//  galaxy
//
//  Created by edy on 2024/6/4.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import AttributedString

enum YHOrderTipsViewStyleType: Int {
    case none = 0
    case onlyMgr = 1
    case onlyOneOrder = 2
    case mgrAndOrder = 3
    
    func getHeight() -> CGFloat {
        switch self {
        case .none:
            return YHOrderTipsView.viewH
        case .onlyMgr, .onlyOneOrder:
            return YHOrderTipsView.viewH2
        case .mgrAndOrder:
            return YHOrderTipsView.viewH3
        }
    }
}

class YHOrderTipsView: UIView {

    fileprivate static let viewH: CGFloat = 84.0
    fileprivate static let viewH2: CGFloat = 154.0
    fileprivate static let viewH3: CGFloat = 218.0
    var buttonClickEvent: (() -> Void)?
    
    var orderTipsViewStyle: YHOrderTipsViewStyleType = .none {
        didSet {
            guard oldValue != orderTipsViewStyle else {
                return
            }
            updateItemsStyle()
        }
    }

    var tipViewH: CGFloat {
        return orderTipsViewStyle.getHeight()
    }
    
    var galaxyManagerItemClickEvent: (() -> Void)?
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
        
        let tap: UITapGestureRecognizer = UITapGestureRecognizer(target: self, action: #selector(tapButton(gestureRecognizer:)))
        addGestureRecognizer(tap)
    }
    
    var dataModelForBasicInfo: YHSignerInfoModel? {
        didSet {
            updateBasicInfo()
        }
    }
    
    var arrLastTwoMessage: [YHHomeLastMessageModel]? {
        didSet {
            updateItems()
            firstItem.message = arrLastTwoMessage?.first
        }
    }
    
    var customerServiceMessage: YHButlerServiceHomeInfoMessage? {
        didSet {
            updateItems()
            mgrItem.updateManagerTipsItemView(message: customerServiceMessage?.serviceMessage?.content ?? "一对一专属服务", didReadMessage: customerServiceMessage?.serviceMessage?.isRead ?? true)
        }
    }
        
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    lazy var bkgImageV: UIImageView = {
        let imgV: UIImageView = UIImageView()
        imgV.contentMode = .scaleAspectFill
        imgV.image = UIImage(named: "home_order_tips_bkg")
        imgV.clipsToBounds = true
        return imgV
    }()
    
    lazy var iconImageV: UIImageView = {
        let imgV: UIImageView = UIImageView()
        imgV.contentMode = .scaleAspectFill
        imgV.image = UIImage(named: "home_order_tips_icon")
        return imgV
    }()
    
    override func layoutSubviews() {
        super.layoutSubviews()
        
        bkgImageV.snp.remakeConstraints { make in
            make.edges.equalToSuperview()
        }
    }
    
    // 智能助理
    lazy var lableTip: UILabel = {
        let label: UILabel = UILabel()
        label.text = "银河管家"
        label.font = UIFont.PFSC_B(ofSize: 15)
        label.textColor = UIColor.mainTextColor
        return label
    }()
    
    // 银河港生活陪伴您的第190天 background: #94A3B8;
    lazy var lableDays: UILabel = {
        let lable: UILabel = UILabel()
        lable.text = "银河港生活陪伴您的第190天"
        lable.font = UIFont.PFSC_R(ofSize: 12)
        lable.textColor = UIColor.yhGreyColor
        return lable
    }()
    
    lazy var firstItem: YHOrderTipsItemView = {
        let itemView: YHOrderTipsItemView = YHOrderTipsItemView()
        itemView.isHidden = true
        return itemView
    }()
    
    lazy var mgrItem: YHGalaxyManagerTipsItemView = {
        let itemView: YHGalaxyManagerTipsItemView = YHGalaxyManagerTipsItemView()
        itemView.isHidden = true
        itemView.buttonClickEvent = { [weak self] in
            self?.galaxyManagerItemClickEvent?()
        }
        return itemView
    }()
    
}

private extension YHOrderTipsView {
    func timeName() -> String {
        let now = Date()
        let hour = Calendar.current.component(.hour, from: now)
        var name = ""
        if hour < 12 {
            name = "上午"
        } else if hour < 18 {
            name = "下午"
        } else {
            name = "晚上"
        }
        return name
    }

    func updateBasicInfo() {
        guard let model = dataModelForBasicInfo else {
            return
        }
                
        let now = Date()
        let timestamp = now.timeIntervalSince1970
        print("当前时间戳：\(timestamp)")
        
        let offsetSecond = CGFloat(Int64(timestamp) - model.registration_timestamp)
        
        if offsetSecond > 0 {
            let roundedUp = Int(ceil(offsetSecond / (24 * 60 * 60)))
            lableDays.text = "银河港生活陪伴您的第" + String(roundedUp) + "天"
        } else {
            lableDays.text = "银河港生活陪伴在您的身边"
        }
        layoutIfNeeded()
    }
    
    private func updateItems() {
        if let mgrMessage = customerServiceMessage, mgrMessage.showButler {
            mgrItem.isHidden = false
        } else {
            mgrItem.isHidden = true
        }
        
        if let arrLastTwoMessage = arrLastTwoMessage, arrLastTwoMessage.count > 0 {
            firstItem.isHidden = false
        } else {
            firstItem.isHidden = true
        }
        if mgrItem.isHidden == false, firstItem.isHidden == false {
            orderTipsViewStyle = .mgrAndOrder
        } else if mgrItem.isHidden == true, firstItem.isHidden == false {
            orderTipsViewStyle = .onlyOneOrder
        } else if mgrItem.isHidden == false, firstItem.isHidden == true {
            orderTipsViewStyle = .onlyMgr
        } else {
            orderTipsViewStyle = .none
        }
        layoutIfNeeded()
        setNeedsLayout()
    }
    
    private func updateItemsStyle() {
        switch orderTipsViewStyle {
        case .none:
            mgrItem.snp.remakeConstraints { make in
                make.top.equalTo(iconImageV.snp.bottom).offset(18)
                make.left.equalTo(kMargin)
                make.width.height.equalTo(0)
            }
            firstItem.snp.remakeConstraints { make in
                make.top.equalTo(mgrItem.snp.bottom).offset(12)
                make.left.equalTo(kMargin)
                make.width.height.equalTo(0)
            }
        case .onlyMgr:
            mgrItem.snp.remakeConstraints { make in
                make.top.equalTo(iconImageV.snp.bottom).offset(18)
                make.left.equalTo(kMargin)
                make.right.equalTo(-kMargin)
                make.height.equalTo(YHGalaxyManagerTipsItemView.viewH)
            }
            firstItem.snp.remakeConstraints { make in
                make.top.equalTo(mgrItem.snp.bottom).offset(12)
                make.left.equalTo(kMargin)
                make.width.height.equalTo(0)
            }
        case .onlyOneOrder:
            mgrItem.snp.makeConstraints { make in
                make.top.equalTo(iconImageV.snp.bottom).offset(18)
                make.left.equalTo(kMargin)
                make.width.height.equalTo(0)
            }
            firstItem.snp.remakeConstraints { make in
                make.top.equalTo(iconImageV.snp.bottom).offset(18)
                make.left.equalTo(kMargin)
                make.right.equalTo(-kMargin)
                make.height.equalTo(YHOrderTipsItemView.viewH)
            }
        case .mgrAndOrder:
            mgrItem.snp.remakeConstraints { make in
                make.top.equalTo(iconImageV.snp.bottom).offset(18)
                make.left.equalTo(kMargin)
                make.right.equalTo(-kMargin)
                make.height.equalTo(YHGalaxyManagerTipsItemView.viewH)
            }
            firstItem.snp.remakeConstraints { make in
                make.top.equalTo(mgrItem.snp.bottom).offset(12)
                make.left.equalTo(kMargin)
                make.right.equalTo(-kMargin)
                make.height.equalTo(YHOrderTipsItemView.viewH)
            }
        }
    }
    
    func setupUI() {
        
        backgroundColor = UIColor.pageBkgColor
        
        clipsToBounds = true
        addSubview(bkgImageV)
        bkgImageV.snp.makeConstraints { make in
            make.left.top.equalToSuperview()
            make.width.equalTo(KScreenWidth - 40)
            make.height.equalTo(YHOrderTipsView.viewH)
        }
        
        // 智能头像
        addSubview(iconImageV)
        iconImageV.snp.makeConstraints { make in
            make.top.equalTo(20)
            make.left.equalTo(kMargin)
            make.width.height.equalTo(44).priority(.high)
        }
        
        addSubview(lableTip)
        lableTip.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(21)
            make.left.equalTo(iconImageV.snp.right).offset(12).priority(.high)
            make.right.lessThanOrEqualToSuperview().offset(-12)
        }
    
        addSubview(lableDays)
        lableDays.snp.makeConstraints { make in
            make.left.equalTo(lableTip.snp.left)
            make.height.equalTo(17)
            make.top.equalTo(lableTip.snp.bottom).offset(4)
        }
        
        addSubview(mgrItem)
        addSubview(firstItem)
        
        layoutIfNeeded()
        setNeedsLayout()
    }
    
    @objc func tapButton(gestureRecognizer: UITapGestureRecognizer) {
        buttonClickEvent?()
    }
    
}
